//------------------------------------------------------------------------------
// File: SnakeValley.cs
// Controls the dead mine scenario tasks
// Author: Matthew Rudge
//------------------------------------------------------------------------------

$SnakeValley::CoreQuestName = "Snake Valley";
$SnakeValley::DryGuyQuestName = "Dry Guy";
$SnakeValley::WaterCost = 20;
$SnakeValley::DryGuyWaterCount = 5;
$SnakeValley::DynamiteDamage = 20;
$SnakeValley::SpearDamage = 50;
$SnakeValley::SasquatchName = "Puddy Tat";

//-Snake Valley Helper Methods--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
//! Activates snake valley tasks
////////////////////////////////////////////////////////////////////////////////
function startSnakeValleyTasks()
{
   InitializeQuestGui();
   tsStartTask("SnakeValley", tsGetStartingVertexes("SnakeValley"));
   tsStartTask("DryGuy", "DryGuyEnterZone");
   tsStartTask("BanditPit", "BanditPitEnterZone");
   tsStartTask("BottleneckPit", "BottleneckEnterZone");
   tsStartTask("SpearPit", "SpearPitZone");
}
////////////////////////////////////////////////////////////////////////////////
//! Deactivates snake valley tasks
////////////////////////////////////////////////////////////////////////////////
function endSnakeValleyTasks()
{
   tsEndTask("SpearPit");
   tsEndTask("BottleneckPit");
   tsEndTask("BanditPit");
   tsEndTask("DryGuy");
   tsEndTask("SnakeValley");
}
////////////////////////////////////////////////////////////////////////////////
//! Win Snake Valley
////////////////////////////////////////////////////////////////////////////////
function winSnakeValley()
{
   // Reward
   tsModifyResource($Resource::Experience, 20, true);
   SendProductionToClient(HeroCharacter, "20 experience");
   // Convert Trading Post
   tsConvertObjTeam("SVTradingpost", $OST_PLAYER);
   
   TaskModifier.strMarkVertex("Campaign2Tasks", "MapSnakeValleyReturn", $TSV_AVAIL);
   //Quest Log
   tsCompleteTaskInTracker("Snake Valley", "Help "@ SVDude.name @".");
}

////////////////////////////////////////////////////////////////////////////////
//! Reset SnakeValley Task
////////////////////////////////////////////////////////////////////////////////
function resetSnakeValleyTasks()
{
   TaskModifier.strResetVertex("Campaign2Tasks", "MapSnakeValleyResetDelay");
   TaskModifier.strMarkVertex("Campaign2Tasks", "MapSnakeValleyResetDelay", $TSV_AVAIL);
}

/******************************************************************************/
/** CORE SNAKE VALLEY TASK
/******************************************************************************/
//-SnakeValleyIntro Vertex------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex initiates the player into the scenario with dialog and task 
// tracker
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyIntro::onEnter(%this) 
{
   Parent::onEnter(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_intro", $TaskButton::Okay, "HeroCharacter");
   tsAddQuestToTracker(true, $SnakeValley::CoreQuestName, "", "Clear out the nest of Copperheads so that the trader can get to his supplies.");
   tsAddQuestToTracker(false, $SnakeValley::DryGuyQuestName, "There sure are a lot of pits around. I certainly wouldn't want to fall into one.", "Find some Water for the Gunslinger stuck in the pit trap.");
   tsReadyQuest($SnakeValley::CoreQuestName);
   tsAddTaskToTracker($SnakeValley::CoreQuestName, "Deal with those snakes.", "Move through the hills to the supply camp the Copperheads have taken over.");
   
   // Complete this vertex
   tsCompleteVertexes("SnakeValley", "SnakeValleyIntro");
}
//-SnakeValleyLose Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Player's game ends once the player has no combat units and the hero is 
// disabled
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyLose::onUpdate(%this)
{
   Parent::onUpdate(%this);
   
   if(HeroCharacter.isHealing && !tsCombatUnitsExist($OST_PLAYER)) {
      tsRestartScenario();
   }
}
//-SnakeValleyWaterZone Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex outputs a message from a civilian when you reach the water zone 
// with the well
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyWaterZone::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_waterman", $TaskButton::Okay, "Waterman");
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_replytowater", $TaskButton::Okay, "HeroCharacter");
}
//-SnakeValleyFenceZone Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex outputs a message when you reach the fence zone
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyFenceZone::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_fence", $TaskButton::Okay, "HeroCharacter");
}
//-SnakeValleyEndZone Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex outputs a message when you reach the end zone
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyEndZone::onExit(%this)
{
   Parent::onExit(%this);
   
   tsCompleteTaskInTracker($SnakeValley::CoreQuestName, "Deal with those snakes.");
   tsSendTaskClickDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_mainreturn", $TaskButton::Accept, "HeroCharacter", "id_snakevalley_mainreturn");
}
//-SnakeValleyReturnToMain Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Returns you to the main map
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyReturnToMain::onExit(%this)
{
   Parent::onExit(%this);
   
   %scneario = new SLScenario() {
      stack = false;
   };
   LoadScenario(%scenario, endSnakeValleyTasks, winSnakeValley, "");
   
   // complete the snake valley satellite
   mapsatellite_snakevalley.disabled = false;
}

/******************************************************************************/
/** MISC BANDIT PIT TASKS
/******************************************************************************/
////////////////////////////////////////////////////////////////////////////////
// This is a bunch of similar vertexes that spawn bandits
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyChokepointSpawn::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Spawn bandits
   %obj = tsCreateObj("Malebandit", "", $OST_NEUTRAL, "-285 -341", "chokepointbandit1");
   %obj.setName("chokepointbandit1");
   %obj = tsCreateObj("Femalebandit", "", $OST_NEUTRAL, "-294 -346", "chokepointbandit2");
   %obj.setName("chokepointbandit2");
}
function SnakeValleyChokepointSpawn::onExit(%this)
{
   Parent::onExit(%this);
   
   tsConvertObjTeam("chokepointbandit1", $OST_BANDIT);
   tsConvertObjTeam("chokepointbandit2", $OST_BANDIT);
}
function SnakeValleyBanditcampSpawn::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Spawn bandits
   %obj = tsCreateObj("Malebandit", "", $OST_NEUTRAL, "-313 -131", "banditcampbandit1");
   %obj.setName("banditcampbandit1");
   %obj = tsCreateObj("Femalebandit", "", $OST_NEUTRAL, "-315 -138", "banditcampbandit2");
   %obj.setName("banditcampbandit2");
}
function SnakeValleyBanditcampSpawn::onExit(%this)
{
   Parent::onExit(%this);
   
   tsConvertObjTeam("banditcampbandit1", $OST_BANDIT);
   tsConvertObjTeam("banditcampbandit2", $OST_BANDIT);
}



/******************************************************************************/
/** DYNAMITE PIT TASKS
/******************************************************************************/
//-SnakeValleyDynamitePit Vertex------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex controls the dynamite pit zone. Once entered, the dynamite object
// is spawned after the hero dialog
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyDynamitePit::onMessage(%this, %msg, %params)
{
   Parent::onMessage(%this, %msg, %params);
   
   if(%this.hasBeenCompleted()) {
      // Send message
      %objs = tsGetObjListFromMsg(%msg, %params);
      %char = %objs.getRandomObject();
      tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_dynamitepit", $TaskButton::Okay, %char);
      
      // Create dynamite
      %obj = tsCreateObj("Dynamite", "BearTrap", $OST_NEUTRAL, "-247 -414");
      
      // Damage objects
      %count = %objs.getSize();
      for(%i = 0; %i < %count; %i++) {
         %char = %objs.getID(%i);
         %char.health -= $SnakeValley::DynamiteDamage;
      }
   }
}
//-SnakeValleyDynamiteExplode Vertex--------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for starting the particle effect on the dynamite
// This MUST happen after the dynamite prop exists on the client (which is why
// there is a slight delay)
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyDynamiteExplode::onExit(%this)
{
   Parent::onExit(%this);
   
   // Destroy dynamite
   tsShowMesh("BearTrap", false);
   tsStartObjParticleFx("BearTrap", "dynamite");
   %count = ClientGroup.getCount();
   for(%i = 0; %i < %count; %i++) {
      %client = ClientGroup.getObject(%i);
      commandToClient(%client, 'PlaySFXSound', audio_dynamiteexplode);
   }
}
//-SnakeValleyDynamiteDestroy Vertex--------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Responsible for destroying the dynamite prop and giving a response dialog
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyDynamiteDestroy::onExit(%this)
{
   Parent::onExit(%this);

   // Response dialog and destruction
   tsRemoveObj("BearTrap");
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_dynamiteboom", $TaskButton::Okay, "HeroCharacter");
}
//-SnakeValleyTntPit Vertex-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex controls the dynamite pit zone. Once entered, the dynamite object
// is spawned
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyTntPit::onMessage(%this, %msg, %params)
{
   Parent::onMessage(%this, %msg, %params);
   
   if(%this.hasBeenCompleted()) {
      // Send message
      %objs = tsGetObjListFromMsg(%msg, %params);
      %char = %objs.getRandomObject();
      
      // Create dynamite
      %obj = tsCreateObj("Dynamite", "TNT", $OST_NEUTRAL, "-125 -226");
      
      // Damage objects
      %count = %objs.getSize();
      for(%i = 0; %i < %count; %i++) {
         %char = %objs.getID(%i);
         %char.health -= $SnakeValley::DynamiteDamage;
      }
   }
}
//-SnakeValleyTntExplode Vertex-------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for starting the particle effect on the dynamite
// This MUST happen after the dynamite prop exists on the client (which is why
// there is a slight delay)
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyTntExplode::onExit(%this)
{
   Parent::onExit(%this);
   
   // Destroy dynamite
   tsShowMesh("TNT", false);
   tsStartObjParticleFx("TNT", "dynamite");
   %count = ClientGroup.getCount();
   for(%i = 0; %i < %count; %i++) {
      %client = ClientGroup.getObject(%i);
      commandToClient(%client, 'PlaySFXSound', audio_dynamiteexplode);
   }
}
//-SnakeValleyTntDestroy Vertex-------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Responsible for destroying the dynamite prop and giving a response dialog
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyTntDestroy::onExit(%this)
{
   Parent::onExit(%this);
   
   // Response dialog and destruction
   tsRemoveObj("TNT");
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_tntboom", $TaskButton::Okay, "HeroCharacter");
}



/******************************************************************************/
/** SASQUATCH PIT TASK
/******************************************************************************/
//-SnakeValleyZilchZone Vertex--------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex monitors when the player enters the sasquatch pit zone
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyZilchZone::onExit(%this)
{
   Parent::onExit(%this);
   
   // Give dialog about entering zone
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_zilch", $TaskButton::Okay, "HeroCharacter", "id_snakevalley_zilch");
   
}
//-SnakeValleySpawnSasquatch Vertex---------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex spawns the sasquatch and begins the idle animation for it
////////////////////////////////////////////////////////////////////////////////
function SnakeValleySpawnSasquatch::onEnter(%this)
{
   Parent::onEnter(%this);
   
   tsCreateObj("Sasquatch", $SnakeValley::SasquatchName, $OST_NEUTRAL, "-298 -294", "putty");
}
function SnakeValleySpawnSasquatch::onExit(%this)
{
   Parent::onExit(%this);
   
   $SnakeValley::SasquatchName.playThread(0, "idle00");
}
//-SnakeValleyMoveSasquatch Vertex----------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for moving the sasquatch away and giving hero
// dialog about the sasquatch
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyMoveSasquatch::onEnter(%this)
{
   // Move sasquatch away
   tsMoveObjToTarget($SnakeValley::SasquatchName, "-450 -300");
   
   // Hero dialog
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_sasquatch", $TaskButton::Okay, "HeroCharacter");
}
function SnakeValleyMoveSasquatch::onUpdate(%this)
{
   %cmpAI = slgQueryInterface($SnakeValley::SasquatchName.getId(), $CID_AI);
   if(!IsFollowingPath(%cmpAI)) {
      $SnakeValley::SasquatchName.DeleteObject();
      tsCompleteVertexes("SnakeValley", "SnakeValleyMoveSasquatch");
   }
}



/******************************************************************************/
/** GOLD BAG PIT TASKS
/******************************************************************************/
////////////////////////////////////////////////////////////////////////////////
// These vertexes monitor and advance the dialog once the player reaches the
// gold bag pit areas and spawn the gold bags in the pits
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyFirstPitZone::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_pitopen", $TaskButton::Okay, "HeroCharacter", "id_snakevalley_pitopen");
}
function SnakeValleyFirstPitOpen::onExit(%this)
{
   Parent::onExit(%this);
   
   // Spawn gold bag
   tsCreateObj("goldbagpickup", "", $OST_NEUTRAL, "-182 -182");
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_pitopentwo", $TaskButton::Okay, "HeroCharacter", "id_snakevalley_pitopentwo");
}
function SnakeValleyArchwayPit::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_archway", $TaskButton::Okay, "HeroCharacter", "id_snakevalley_archway");
}
function SnakeValleyArchwayOpen::onExit(%this)
{
   Parent::onExit(%this);

   // Spawn gold bag
   tsCreateObj("goldbagpickup", "", $OST_NEUTRAL, "-338 -331");
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_archwayopen", $TaskButton::Okay, "HeroCharacter");
}
function SnakeValleyBeginLeftZone::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_beginleft", $TaskButton::Okay, "HeroCharacter", "id_snakevalley_beginleft");
}
function SnakeValleyBeginLeftOpen::onExit(%this)
{
   Parent::onExit(%this);
   
   // Spawn gold bag
   tsCreateObj("goldbagpickup", "", $OST_NEUTRAL, "-138 -182");
}


/******************************************************************************/
/** GOLD BAG PIT TASKS
/******************************************************************************/
//-SnakeValleyWoodZone Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Monitors when the player enters the "wood" zone and sends a pop up dialog
// when it happens
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyWoodZone::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_snakevalley_wood", $TaskButton::Okay, "HeroCharacter", "id_snakevalley_wood");
}
//-SnakeValleyWoodDialogDone Vertex---------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Spawns the wood prop after the dialog is finished
////////////////////////////////////////////////////////////////////////////////
function SnakeValleyWoodDialogDone::onExit(%this)
{
   Parent::onExit(%this);
   
   // Spawn wood prop
   tsCreateObj("woodpickup", "", "", "-157.5 -342.5");
   tsCompleteVertexes("SnakeValley", "SnakeValleyWoodDialogDone");
}



/******************************************************************************/
/** BOTTLENECK TASK
/******************************************************************************/
//-BottleneckEnterZone Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Starts dialog once player enters "bottleneck zone
////////////////////////////////////////////////////////////////////////////////
function BottleneckEnterZone::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_bottleneck_pitopen", $TaskButton::Okay, "HeroCharacter", "id_bottleneck_pitopen");
}

//-BottleneckChest Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Spawns the chest that can be picked up and adds dynamite to the player's
// inventory. If chest is picked up, two bandits are spawned
////////////////////////////////////////////////////////////////////////////////
function BottleneckChest::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Spawn gold chest
   tsCreateObj("chest", "", "", "-201.5 -330.5", "bottleneckchest");
   
   // Add dynamite to inventory
   %client = ClientGroup.getObject(0);
   %slot = invGetSlotForItem(%client, $INV_DYNAMITE);
   invPushItemInInventory(%client, %slot);
}
function BottleneckChest::onExit(%this)
{
   Parent::onExit(%this);
   
   // Spawn bandits
   tsCreateObj("Malebandit", "", $OST_BANDIT, "-214 -331", "bottlebandit1");
   %obj = tsCreateObj("Femalebandit", "", $OST_BANDIT, "-229 -317", "bottlebandit2");
   %obj.setName("bottlebandit2");
   
}
//-BottleneckBandits Vertex-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// Responsible for dialog from bandits after they have been created
////////////////////////////////////////////////////////////////////////////////
function BottleneckBandits::onExit(%this)
{
   Parent::onExit(%this);
   
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_bottleneck_bandits", $TaskButton::Okay, "bottlebandit2");
}



/******************************************************************************/
/** DRY GUY TASK
/******************************************************************************/
//-DryGuyEnterZone Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex monitors when the player first enters the "dryguy" zone. Once 
// this happens, the dryguy is spawned in the pit
////////////////////////////////////////////////////////////////////////////////
function DryGuyEnterZone::onExit(%this)
{
   Parent::onExit(%this);
   
   // Spawn dryguy
   %obj = tsCreateObj("Malegunslinger", "", $OST_NEUTRAL, "-109.5 -312.5", "dryguy");
   %obj.setName("dryguy");
   %obj.showMesh(false);
}
//-DryGuyDialog Vertex----------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex will spawn the dialog between the player and dryguy and add
// a task to the tracker for getting water for the dryguy
////////////////////////////////////////////////////////////////////////////////
function DryGuyDialog::onExit(%this)
{
   Parent::onExit(%this);
   
   // Dialog between hero and dryguy
   tsSendTaskPopUpDlg("ID_TASKNAME_DRYGUY", "id_dryguy_dryguyparch", $TaskButton::Okay, "dryguy");
   tsSendTaskPopUpDlg("ID_TASKNAME_DRYGUY", "id_dryguy_herotodryguy", $TaskButton::Okay, "HeroCharacter");
   %dlgData = tsCreateTaskDlgData("ID_TASKNAME_DRYGUY", true, $TaskButton::Accept, "id_dryguy_needwater");
   tsAddAcceptCost($Resource::Water, $SnakeValley::DryGuyWaterCount, %dlgData);
   tsAddTaskDialog("id_dryguy_needwater", %dlgData);
   tsSendTaskDlgMsg(%dlgData, "dryguy");
   tsDestroyTaskDlgData(%dlgData);
   
   // Task log
   tsReadyQuest($SnakeValley::DryGuyQuestName);
   tsAddTaskToTracker($SnakeValley::DryGuyQuestName, "Get " @ dryguy.name @ " 5 Water.", "Get " @ dryguy.name @ " 5 canteens of Water.");
}
//-DryGuyNeedWaterWait Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex will be executed after two seconds have elapsed once the player
// enters the needwater zone for the dryguy. It is responsible for the dialog
// between the water man and the hero concerning the water
////////////////////////////////////////////////////////////////////////////////
function DryGuyNeedWaterWait::onExit(%this)
{
   Parent::onExit(%this);
   
   // Send dialog between hero and waterman
   tsSendTaskPopUpDlg("ID_TASKNAME_DRYGUY", "id_dryguy_herofindwater", $TaskButton::Okay, "HeroCharacter");
   %dlgData = tsCreateTaskDlgData("ID_TASKNAME_DRYGUY", true, $TaskButton::Accept, "id_dryguy_waterprice");
   tsAddAcceptCost($Resource::Gold, $SnakeValley::WaterCost, %dlgData);
   tsAddTaskDialog("id_dryguy_waterprice", %dlgData);
   tsSendTaskDlgMsg(%dlgData, "Waterman");
   tsDestroyTaskDlgData(%dlgData);
}
//-DryGuyPayForWater Vertex-----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is completed once the player accepts the gold cost to obtain the
// water for the dry guy. It is responsible for deducting and adding the 
// resources
////////////////////////////////////////////////////////////////////////////////
function DryGuyPayForWater::onExit(%this)
{
   // Waterman's dialog
   tsSendTaskPopUpDlg("ID_TASKNAME_DRYGUY", "id_dryguy_buywater", $TaskButton::Okay, "Waterman");
   
   // Deduct and add resources
   tsModifyResource($Resource::Gold, $SnakeValley::WaterCost, false);
   %cap = rsGetResourceCap($Resource::Water);
   if(%cap < 5) {
      tsModifyResourceMax($Resource::Water, $SnakeValley::DryGuyWaterCount - %cap, true);
   }
   tsModifyResource($Resource::Water, $SnakeValley::DryGuyWaterCount, true);
   
}
//-DryGuyGiveWater Vertex-------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex completes once the dry guy gets his water from the player. It
// is responsible for the dry guy "thank" dialog, as well as task log update
// and making dry guy visible and part of player team
////////////////////////////////////////////////////////////////////////////////
function DryGuyGiveWater::onExit(%this)
{
   // Dry guy "thank" dialog
   tsSendTaskPopUpDlg("ID_TASKNAME_DRYGUY", "id_dryguy_getwater", $TaskButton::Okay, "dryguy");
   
   // Update task log
   tsCompleteTaskInTracker($SnakeValley::DryGuyQuestName, "Get " @ dryguy.name @ " 5 Water.");
   
   // Update dry guy to player's team
   tsShowMesh("dryguy", true);
   tsConvertObjTeam("dryguy", $OST_PLAYER);
}



/******************************************************************************/
/** BANDIT PIT TASK
/******************************************************************************/
//-BanditPitEnterZone Vertex----------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex shoots out dialog from the hero character once the bandit pit
// zone has been reached, including the spawning of two of the three bandits
////////////////////////////////////////////////////////////////////////////////
function BanditPitEnterZone::onExit(%this)
{
   Parent::onExit(%this);
   
   // Dialog from hero character
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_bandpit_pitopen", $TaskButton::Okay, "HeroCharacter");
   
   // Spawn bandits
   %obj = tsCreateObj("Malebandit", "", $OST_NEUTRAL, "-333 -376", "pitbandit1");
   %obj.setName("pitbandit1");
   %obj = tsCreateObj("Femalebandit", "", $OST_NEUTRAL, "-323 -373", "pitbandit2");
   %obj.setName("pitbandit2");
}
//-BanditPitWhereVelma Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for giving dialog from one of the bandits asking
// where Velma is (she's a little slow)
////////////////////////////////////////////////////////////////////////////////
function BanditPitWhereVelma::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Send dialog about velma
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_bandpit_surprise", $TaskButton::Okay, "pitbandit1", "id_bandpit_surprise");
   
}
//-BanditPitSpawnVelma Vertex---------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex is responsible for spawning Velma
////////////////////////////////////////////////////////////////////////////////
function BanditPitSpawnVelma::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Spawn Velma
   %obj = tsCreateObj("Femalebandit", "Velma", $OST_NEUTRAL, "-329 -367", "pitbandit3");
   %obj.setName("pitbandit3");
}
//-BanditPitBanditDefeat Vertex-------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This vertex completes once the bandits have been defeated. Upon entering this
// vertex, the bandits teams are changed to hostile
////////////////////////////////////////////////////////////////////////////////
function BanditPitBanditDefeat::onEnter(%this)
{
   Parent::onEnter(%this);
   
   // Switch bandit teams
   tsConvertObjTeam("pitbandit1", $OST_BANDIT);
   tsConvertObjTeam("pitbandit2", $OST_BANDIT);
   tsConvertObjTeam("pitbandit3", $OST_BANDIT);
}
function BanditPitBanditDefeat::onExit(%this)
{
   Parent::onExit(%this);
   
   // End dialog
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_bandpit_defeat", $TaskButton::Okay, "HeroCharacter");
}



/******************************************************************************/
/** SPEAR PIT TASK
/******************************************************************************/
//-SpearPitZone Vertex----------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This spawns the spears when a character enters the zone and damages all units
// in the zone
////////////////////////////////////////////////////////////////////////////////
function SpearPitZone::onMessage(%this, %msg, %params)
{
   Parent::onMessage(%this, %msg, %params);
   
   if(%this.hasBeenCompleted()) {
      // Spawn spears
      tsCreateObj("spear", "", "", "-359 -180");
      tsCreateObj("spear", "", "", "-361 -180");
      tsCreateObj("spear", "", "", "-363 -179");
      tsCreateObj("spear", "", "", "-363 -182");
      tsCreateObj("spear", "", "", "-364 -183");
      tsCreateObj("spear", "", "", "-363 -185");
      tsCreateObj("spear", "", "", "-361 -184");
      tsCreateObj("spear", "", "", "-359 -185");
      tsCreateObj("spear", "", "", "-359 -182");
      
      // Damage units
      %objs = tsGetObjListFromMsg(%msg, %params);
      %count = %objs.getSize();
      for(%i = 0; %i < %count; %i++) {
         %char = %objs.getID(%i);
         %char.health -= $SnakeValley::SpearDamage;
      }
   }
}
//-SpearPitDialog Vertex--------------------------------------------------------
////////////////////////////////////////////////////////////////////////////////
// This initiates the dialog after a short delay
////////////////////////////////////////////////////////////////////////////////
function SpearPitDialog::onExit(%this)
{
   tsSendTaskPopUpDlg("ID_TASKNAME_SNAKEVALLEY", "id_spearpit_response", $TaskButton::Okay, "HeroCharacter");
}

// End SnakeValley.cs